"use client";

import { useEffect, useState } from "react";

import FormModal from "@/components/FormModal";
import Image from "next/image";
import Link from "next/link";
import Pagination from "@/components/Pagination";
import Table from "@/components/Table";
import TableSearch from "@/components/TableSearch";
import { getCellGroups } from "../../../../../lib/api";

type CellGroup = {
  groupId: number;
  description: string;
  group: string;
  leader: string[];
  meetingDay: string;
  time: string;
  venue: string;
};

const columns = [
  { header: "Group Name", accessor: "group" },
  {
    header: "Group No.",
    accessor: "groupId",
    className: "hidden md:table-cell",
  },
  {
    header: "Description",
    accessor: "description",
    className: "hidden md:table-cell",
  },
  {
    header: "Group Leader",
    accessor: "leader",
    className: "hidden lg:table-cell",
  },
  {
    header: "Meeting Day",
    accessor: "meetingDay",
    className: "hidden md:table-cell",
  },
  { header: "Time", accessor: "time", className: "hidden md:table-cell" },
  { header: "Venue", accessor: "venue", className: "hidden md:table-cell" },
  { header: "Actions", accessor: "actions" },
];

const CellGroupPage = () => {
  const [cellGroups, setCellGroups] = useState<CellGroup[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [role, setRole] = useState<string>("");

  useEffect(() => {
    const fetchGroups = async () => {
      try {
        const data = await getCellGroups();
        setCellGroups(data);
      } catch (error) {
        console.error("Error fetching cell groups:", error);
      } finally {
        setLoading(false);
      }
    };

    fetchGroups();

    // Mock role fetching (Replace with actual authentication logic)
    setRole(localStorage.getItem("role") || "user");
  }, []);

  const renderRow = (item: CellGroup) => (
    <tr
      key={item.groupId}
      className="border-b border-gray-200 even:bg-slate-50 text-sm hover:bg-slate-50"
    >
      <td className="flex items-center gap-4 p-4">
        <div className="flex flex-col">
          <h3 className="font-semibold">{item.group}</h3>
        </div>
      </td>
      <td className="hidden md:table-cell">{item.groupId}</td>
      <td className="hidden md:table-cell">{item.description}</td>
      <td className="hidden md:table-cell">{item.leader.join(", ")}</td>
      <td className="hidden md:table-cell">{item.meetingDay}</td>
      <td className="hidden md:table-cell">{item.time}</td>
      <td className="hidden md:table-cell">{item.venue}</td>
      <td>
        <div className="flex items-center gap-2">
          <Link href={`/dashboard/list/cellGroups/${item.groupId}`}>
            <button className="w-8 h-8 flex items-center justify-center rounded-full bg-white">
              <Image src="/view.png" alt="" width={16} height={16} />
            </button>
          </Link>
          {role === "admin" && (
            <>
              <FormModal table="CellGroup" type="update" data={item} />
              <FormModal table="CellGroup" type="delete" id={item.groupId} />
            </>
          )}
        </div>
      </td>
    </tr>
  );

  return (
    <div className="bg-white p-4 rounded-md flex-1 m-4 mt-4">
      {/* TOP */}
      <div className="flex items-center justify-between">
        <h1 className="hidden md:block text-lg font-semibold">All Groups</h1>
        <div className="flex flex-col md:flex-row items-center gap-4 w-full md:w-auto">
          <TableSearch />
          <div className="flex items-center gap-4 self-end">
            <button className="w-6 h-6 flex items-center justify-center rounded-full bg-red-300">
              <Image src="/filter.png" alt="Filter" width={14} height={14} />
            </button>
            <button className="w-6 h-6 flex items-center justify-center rounded-full bg-red-300">
              <Image src="/sort.png" alt="Sort" width={14} height={14} />
            </button>
            {role === "admin" && <FormModal table="CellGroup" type="create" />}
          </div>
        </div>
      </div>

      {/* LIST */}
      {loading ? (
        <p className="text-center text-gray-500">Loading cell groups...</p>
      ) : (
        <Table columns={columns} renderRow={renderRow} data={cellGroups} />
      )}
    </div>
  );
};

export default CellGroupPage;
