"use client";

import { useEffect, useState } from "react";

import FormModal from "@/components/FormModal";
import Image from "next/image";
import Link from "next/link";
import Pagination from "@/components/Pagination";
import Table from "@/components/Table";
import TableSearch from "@/components/TableSearch";
import { getInquiries } from "../../../../../lib/api";

type Inquiry = {
  inquiryId: number;
  memberName: string;
  memberId: string;
  email: string;
  mobNo: string[];
  inquiryType: string;
  subject: string;
  status: string;
};

const columns = [
  { header: "Inquiry", accessor: "inquiryId" },
  { header: "Member Name", accessor: "memberName" },
  {
    header: "Member ID",
    accessor: "memberId",
    className: "hidden md:table-cell",
  },
  { header: "Email", accessor: "email", className: "hidden lg:table-cell" },
  {
    header: "Mobile No.",
    accessor: "mobNo",
    className: "hidden md:table-cell",
  },
  {
    header: "Inquiry",
    accessor: "inquiryType",
    className: "hidden md:table-cell",
  },
  { header: "Subject", accessor: "subject", className: "hidden md:table-cell" },
  { header: "Status", accessor: "status", className: "hidden md:table-cell" },
  { header: "Actions", accessor: "actions" },
];

const MemberPage = () => {
  const [inquiries, setInquiries] = useState<Inquiry[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [role, setRole] = useState<string>("");

  useEffect(() => {
    const fetchInquiries = async () => {
      try {
        const data = await getInquiries();
        setInquiries(data);
      } catch (error) {
        console.error("Error fetching inquiries:", error);
      } finally {
        setLoading(false);
      }
    };

    fetchInquiries();
    setRole(localStorage.getItem("role") || "user");
  }, []);

  const renderRow = (item: Inquiry) => (
    <tr
      key={item.inquiryId}
      className="border-b border-gray-200 even:bg-slate-50 text-sm hover:bg-slate-50"
    >
      <td className="p-4">{item.inquiryId}</td>
      <td className="hidden md:table-cell">{item.memberName}</td>
      <td className="hidden md:table-cell">{item.memberId}</td>
      <td className="hidden md:table-cell">{item.email}</td>
      <td className="hidden md:table-cell">{item.mobNo.join(", ")}</td>
      <td className="hidden md:table-cell">{item.inquiryType}</td>
      <td className="hidden md:table-cell">{item.subject}</td>
      <td className="hidden md:table-cell">{item.status}</td>
      <td>
        <div className="flex items-center gap-2">
          <Link href={`/dashboard/list/inquiries/${item.inquiryId}`}>
            <button className="w-8 h-8 flex items-center justify-center rounded-full bg-white">
              <Image src="/view.png" alt="" width={16} height={16} />
            </button>
          </Link>
          {role === "admin" && (
            <>
              <FormModal table="Inquiries" type="update" data={item} />
              <FormModal table="Inquiries" type="delete" id={item.inquiryId} />
            </>
          )}
        </div>
      </td>
    </tr>
  );

  return (
    <div className="bg-white p-4 rounded-md flex-1 m-4 mt-4">
      {/* TOP */}
      <div className="flex items-center justify-between">
        <h1 className="hidden md:block text-lg font-semibold">
          General Inquiries
        </h1>
        <div className="flex flex-col md:flex-row items-center gap-4 w-full md:w-auto">
          <TableSearch />
          <div className="flex items-center gap-4 self-end">
            <button className="w-6 h-6 flex items-center justify-center rounded-full bg-red-300">
              <Image src="/filter.png" alt="Filter" width={14} height={14} />
            </button>
            <button className="w-6 h-6 flex items-center justify-center rounded-full bg-red-300">
              <Image src="/sort.png" alt="Sort" width={14} height={14} />
            </button>
            {role === "admin" && <FormModal table="Inquiries" type="create" />}
          </div>
        </div>
      </div>

      {/* LIST */}
      {loading ? (
        <p className="text-center text-gray-500">Loading inquiries...</p>
      ) : (
        <Table columns={columns} renderRow={renderRow} data={inquiries} />
      )}
    </div>
  );
};

export default MemberPage;
