"use client";

import { useEffect, useState } from "react";

import FormModal from "@/components/FormModal";
import Image from "next/image";
import Link from "next/link";
import Pagination from "@/components/Pagination";
import Table from "@/components/Table";
import TableSearch from "@/components/TableSearch";
import { getPrayers } from "../../../../../lib/api";

type Prayers = {
  requestId: number;
  memberName: string;
  description: string;
  requestStatus: string;
};

const columns = [
  { header: "Prayer Request", accessor: "requestId" },
  { header: "Member Name", accessor: "memberName" },

  {
    header: "Description",
    accessor: "description",
    className: "hidden md:table-cell",
  },
  {
    header: "Status",
    accessor: "requestStatus",
    className: "hidden lg:table-cell",
  },
  { header: "Actions", accessor: "actions" },
];

const PrayerRequestPage = () => {
  const [prayers, setPrayers] = useState<Prayers[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);
  const [role, setRole] = useState<string | null>(null);

  useEffect(() => {
    const fetchPrayers = async () => {
      try {
        const data = await getPrayers();
        setPrayers(data);
        console.log(data);
      } catch (error: any) {
        setError(error.message);
      } finally {
        setLoading(false);
      }
    };

    fetchPrayers();
    setRole(localStorage.getItem("role") || "guest"); // Fetch role from localStorage
  }, []);

  const renderRow = (item: Prayers) => (
    <tr
      key={item.requestId}
      className="border-b border-gray-200 even:bg-slate-50 text-sm hover:bg-slate-50"
    >
      <td className="flex items-center gap-4 p-4">
        <div className="flex flex-col">
          <h3 className="font-semibold">{item.requestId}</h3>
        </div>
      </td>
      <td className="hidden md:table-cell">{item.memberName}</td>

      <td className="hidden md:table-cell">{item.description}</td>
      <td className="hidden md:table-cell">{item.requestStatus}</td>
      <td>
        <div className="flex items-center gap-2">
          <Link href={`/dashboard/list/prayers/${item.requestId}`}>
            <button className="w-8 h-8 flex items-center justify-center rounded-full bg-white">
              <Image src="/view.png" alt="" width={16} height={16} />
            </button>
          </Link>
          {role === "admin" && (
            <>
              <FormModal table="PrayerRequest" type="update" data={item} />
              <FormModal
                table="PrayerRequest"
                type="delete"
                id={item.requestId}
              />
            </>
          )}
        </div>
      </td>
    </tr>
  );

  return (
    <div className="bg-white p-4 rounded-md flex-1 m-4 mt-4">
      {/* TOP */}
      <div className="flex items-center justify-between">
        <h1 className="hidden md:block text-lg font-semibold">
          All Prayer Requests
        </h1>
        <div className="flex flex-col md:flex-row items-center gap-4 w-full md:w-auto">
          <TableSearch />
          <div className="flex items-center gap-4 self-end">
            <button className="w-6 h-6 flex items-center justify-center rounded-full bg-red-300">
              <Image src="/filter.png" alt="Filter" width={14} height={14} />
            </button>
            <button className="w-6 h-6 flex items-center justify-center rounded-full bg-red-300">
              <Image src="/sort.png" alt="Sort" width={14} height={14} />
            </button>
            {role === "admin" && (
              <FormModal table="PrayerRequest" type="create" />
            )}
          </div>
        </div>
      </div>

      {/* LIST */}
      {loading ? (
        <p>Loading prayer requests...</p>
      ) : error ? (
        <p className="text-red-500">{error}</p>
      ) : (
        <Table columns={columns} renderRow={renderRow} data={prayers} />
      )}
    </div>
  );
};

export default PrayerRequestPage;
