"use client";
import React from "react";
import { zodResolver } from "@hookform/resolvers/zod";
import { useForm } from "react-hook-form";
import { z } from "zod";
import InputField from "@/components/InputField";
import Image from "next/image";

const schema = z.object({
  username: z
    .string()
    .min(3, { message: "Username must be at least 3 characters long!" })
    .max(20, { message: "Username must be at most 20 characters long!" }),
  email: z.string().email({ message: "Invalid email address!" }),
  password: z
    .string()
    .min(8, { message: "Password must be at least 8 characters long!" }),
  inquiry: z.string().min(1, { message: "Inquiry is required!" }),
  inquiryId: z.string().min(1, { message: "Inquiry ID is required!" }),
  subject: z.string().min(1, { message: "Subject is required!" }),
  memberName: z.string().min(1, { message: "Member Name is required!" }),
  phone: z.string().min(1, { message: "Phone is required!" }),
  emailaddress: z.string().min(1, { message: "Email of the member is required!" }),
  status: z.string().min(1, { message: "The status of the inquiry is required!" }),
});

type Inputs = z.infer<typeof schema>;

const InquiryForm = ({
  type,
  data,
}: {
  type: "create" | "update";
  data?: any;
}) => {
  const {
    register,
    handleSubmit,
    formState: { errors },
  } = useForm<Inputs>({
    resolver: zodResolver(schema),
    defaultValues :{
      username: type === "update" ? data?.username : "",
      email: type === "update" ? data?.email : "",
      password: type === "update" ? data?.password : "",
      inquiry: type === "update" ? data?.inquiry : "",
      inquiryId: type === "update" ? data?.inquiryId : "",
      subject: type === "update" ? data?.subject : "",
      memberName: type === "update" ? data?.memberName : "",
      phone: type === "update" ? data?.phone : "",
      emailaddress: type === "update" ? data?.emailaddress : "",
      status: type === "update" ? data?.status : "open", 
    },
  });

  const onSubmit = handleSubmit((data) => {
    console.log(data);
  });


  return (
    <form className="flex flex-col gap-8" onSubmit={onSubmit}>
      <h1 className="text-xl font-semibold">Create a new inquiry</h1>
      <span className="text-xs text-gray-400 font-medium">
        Authentication Information
      </span>
      <div className="flex justify-between flex-wrap gap-4">
        <InputField
          label="Username"
          name="username"
          defaultValue={data?.username}
          register={register}
          error={errors?.username}
        />
        <InputField
          label="Email"
          name="email"
          defaultValue={data?.email}
          register={register}
          error={errors?.email}
        />
        <InputField
          label="Password"
          name="password"
          type="password"
          defaultValue={data?.password}
          register={register}
          error={errors?.password}
        />
      </div>
      <span className="text-xs text-gray-400 font-medium">
        Personal Information
      </span>
      <div className="flex justify-between flex-wrap gap-4">
        <InputField
          label="Inquiry"
          name="inquiry"
          defaultValue={data?.inquiry}
          register={register}
          error={errors.inquiry}
        />
        <InputField
          label="Inquiry ID"
          name="inquiryId"
          defaultValue={data?.inquiryId}
          register={register}
          error={errors.inquiryId}
        />
        <InputField
          label="Subject"
          name="subject"
          defaultValue={data?.subject}
          register={register}
          error={errors.subject}
        />
        <InputField
          label="Name of Member"
          name="memberName"
          defaultValue={data?.memberName}
          register={register}
          error={errors.memberName}
        />
        <InputField
          label="Phone"
          name="phone"
          defaultValue={data?.phone}
          register={register}
          error={errors.phone}
        />
        <InputField
          label="Email"
          name="emailaddress"
          defaultValue={data?.emailaddress}
          register={register}
          error={errors.emailaddress}
          type="emailaddress"
        />
        <InputField
          label="Status"
          name="status"
          defaultValue={data?.emailaddress}
          register={register}
          error={errors.emailaddress}
          type="status"
          />
      </div>
      <button className="bg-blue-400 text-white p-2 rounded-md hover:bg-blue-700 transition">
        {type === "create" ? "Create" : "Update"}
      </button>
    </form>
  );
};

export default InquiryForm;