"use client";

import { useEffect, useState } from "react";

import FormModal from "@/components/FormModal";
import Image from "next/image";
import { getVisitorById } from "../../lib/api";

type Props = {
  visitorId: number;
};

const SingleVisitorPage = ({ visitorId }: Props) => {
  const [visitorData, setVisitorData] = useState<any | null>(null);

  useEffect(() => {
    const fetchVisitorData = async () => {
      try {
        const data = await getVisitorById(visitorId);
        setVisitorData(data);
      } catch (error) {
        console.error("Error fetching visitor data:", error);
      }
    };

    fetchVisitorData();
  }, [visitorId]);

  if (!visitorData) {
    return <div>Loading...</div>;
  }

  return (
    <div className="flex-1 p-4 flex flex-col gap-4 xl:flex-row w-full">
      <div className="w-full">
        {/* TOP */}
        <div className="flex flex-col lg:flex-row gap-4 w-full">
          <div className="bg-slate-400 py-3 px-4 rounded-md flex gap-4 w-full h-48 mt-2">
            <div className="w-1/3">
              <Image
                src="/avatar.png"
                alt=""
                width={100}
                height={100}
                className="w-36 h-36 rounded-full object-cover mt-3"
              />
            </div>
            <div className="w-2/3 flex flex-col justify-between gap-2">
              <div className="flex items-center gap-4">
                <h1 className="text-xl font-semibold">{visitorData.name}</h1>
                <FormModal table="Visitors" type="update" />
              </div>
              <div className="flex items-center justify-between gap-2 flex-wrap text-xs font-medium">
                <div className="w-full md:w-1/3 flex items-center gap-2">
                  <Image src="/role.png" alt="" width={18} height={18} />
                  <span className="text-lg font-medium text-gray-700 mt-0">
                    {visitorData.role || "Visitor"}
                  </span>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* PERSONAL INFORMATION */}
        <div className="mt-4 bg-slate-300 py-6 px-4 rounded-md w-full">
          <h2 className="text-lg font-semibold mb-4">Personal Information</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            <InfoBox icon="/gender.png" label={visitorData.gender} />
            <InfoBox icon="/birthday.png" label={visitorData.dob} />
            <InfoBox icon="/mobile.png" label={visitorData.phoneNumber} />
            <InfoBox icon="/email.png" label={visitorData.email} />
            <InfoBox icon="/id-card.png" label={visitorData.idNumber} />
            <InfoBox icon="/resident.png" label={visitorData.address} />
          </div>
        </div>

        {/* VISITOR INFORMATION */}
        <div className="mt-4 bg-slate-300 py-6 px-4 rounded-md w-full">
          <h2 className="text-lg font-semibold mb-4">Visitor Information</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            <InfoBox icon="/date_joined.png" label={visitorData.dateJoined} />
            <InfoBox icon="/status.png" label={visitorData.membershipStatus} />
            <InfoBox
              icon="/events.png"
              label={visitorData.baptismStatus || "Not baptized"}
            />
          </div>
        </div>

        {/* NOTES */}
        <div className="bg-white p-4 rounded-lg shadow flex flex-col gap-2 col-span-full mt-4">
          <h3 className="text-md font-semibold">Notes</h3>
          <textarea
            name="notes"
            className="w-full p-2 border rounded-md resize-none focus:outline-none focus:ring-2 focus:ring-blue-400"
            rows={4}
            placeholder="Leave a note about this visitor..."
            defaultValue={visitorData.notes || ""}
          />
        </div>
      </div>
    </div>
  );
};

// Helper Component for displaying info items
const InfoBox = ({ icon, label }: { icon: string; label: string }) => (
  <div className="bg-white p-4 rounded-lg shadow flex items-center gap-4">
    <Image src={icon} alt="" width={18} height={18} />
    <span className="font-medium">{label || "Not provided"}</span>
  </div>
);

export default SingleVisitorPage;
